import { prisma } from '@/lib/prisma'
import { NextResponse } from 'next/server'

// PUT /api/users/[id] - Update user
export async function PUT(
    request: Request,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params
        const body = await request.json()
        const { nama, email, role, noWa, targetPsbMonthly, targetVisitDaily } = body

        const user = await prisma.user.update({
            where: { id },
            data: {
                ...(nama && { nama }),
                ...(email && { email }),
                ...(role && { role }),
                ...(noWa && { noWa }),
                ...(targetPsbMonthly !== undefined && { targetPsbMonthly }),
                ...(targetVisitDaily !== undefined && { targetVisitDaily }),
            },
            select: {
                id: true,
                nama: true,
                email: true,
                role: true,
                noWa: true,
                isActive: true,
                targetPsbMonthly: true,
                targetVisitDaily: true,
                createdAt: true,
            },
        })

        return NextResponse.json(user)
    } catch (error) {
        console.error('PUT /api/users/[id] error:', error)
        return NextResponse.json({ error: 'Failed to update user' }, { status: 500 })
    }
}

// PATCH /api/users/[id] - Toggle active status
export async function PATCH(
    request: Request,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params

        const current = await prisma.user.findUnique({ where: { id }, select: { isActive: true } })
        if (!current) {
            return NextResponse.json({ error: 'User not found' }, { status: 404 })
        }

        const user = await prisma.user.update({
            where: { id },
            data: { isActive: !current.isActive },
            select: {
                id: true,
                nama: true,
                email: true,
                role: true,
                noWa: true,
                isActive: true,
                targetPsbMonthly: true,
                targetVisitDaily: true,
                createdAt: true,
            },
        })

        return NextResponse.json(user)
    } catch (error) {
        console.error('PATCH /api/users/[id] error:', error)
        return NextResponse.json({ error: 'Failed to toggle user status' }, { status: 500 })
    }
}
